# 🔧 "Access Denied for User" - Troubleshooting Guide

## ⚠️ The Error You're Seeing
```
Access denied for user 'root'@'localhost'
```

This means PHP can't connect to your MySQL database. Here's how to fix it:

---

## 🔍 Step 1: Check if MySQL is Running

### On Windows:
```powershell
# Check if MySQL service is running
Get-Service -Name "MySQL80" | Select-Object Status

# If not running, start it
Start-Service -Name "MySQL80"
```

### Common MySQL Service Names:
- `MySQL80` (MySQL 8.0)
- `MySQL57` (MySQL 5.7)
- `MariaDB` (MariaDB Server)

---

## 🛠️ Step 2: Verify Your Database Credentials

### Option A: Default Setup (Most Common)
If you just installed MySQL with default settings:

**Edit:** `includes/config.php`

```php
define('DB_HOST', 'localhost');
define('DB_USER', 'root');
define('DB_PASS', '');           // Empty password
define('DB_NAME', 'hotel_db');
```

### Option B: Custom Setup
If you set a password during MySQL installation:

```php
define('DB_HOST', 'localhost');
define('DB_USER', 'root');
define('DB_PASS', 'your_password_here');  // Your MySQL root password
define('DB_NAME', 'hotel_db');
```

---

## 📊 Step 3: Create Database & Import Schema

### Using MySQL Command Line:

1. **Open Command Prompt as Administrator** and connect to MySQL:
```cmd
mysql -u root -p
```

2. **When prompted, enter your password** (press Enter if no password)

3. **Run these commands:**
```sql
-- Create database
CREATE DATABASE IF NOT EXISTS hotel_db;

-- Use the database
USE hotel_db;

-- Now import the schema
-- Copy the contents of database/hotel_db.sql and paste here
```

### Using MySQL Workbench (Easier):

1. Open **MySQL Workbench**
2. Connect to your MySQL server
3. Go to **Server** → **Data Import**
4. Select **Import from Self-Contained File**
5. Choose `database/hotel_db.sql`
6. Click **Start Import**

### Using Command Line with SQL File:
```cmd
mysql -u root -p < database\hotel_db.sql
```

---

## 🔐 Step 4: Check/Create MySQL User

If you want a specific user (recommended for production):

```sql
-- Create new user
CREATE USER 'hotel_user'@'localhost' IDENTIFIED BY 'hotel_password';

-- Grant all permissions on hotel_db
GRANT ALL PRIVILEGES ON hotel_db.* TO 'hotel_user'@'localhost';

-- Apply changes
FLUSH PRIVILEGES;
```

Then update `includes/config.php`:
```php
define('DB_USER', 'hotel_user');
define('DB_PASS', 'hotel_password');
```

---

## ✅ Step 5: Test Connection

Create a test file: `test_connection.php`

```php
<?php
define('DB_HOST', 'localhost');
define('DB_USER', 'root');
define('DB_PASS', '');
define('DB_NAME', 'hotel_db');

try {
    $pdo = new PDO(
        'mysql:host=' . DB_HOST . ';dbname=' . DB_NAME,
        DB_USER,
        DB_PASS,
        [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
    );
    echo "✅ Connection successful!";
    
    // List tables
    $result = $pdo->query("SHOW TABLES");
    echo "<br>Tables found: " . $result->rowCount();
    
} catch (PDOException $e) {
    echo "❌ Connection failed: " . $e->getMessage();
}
?>
```

**Access:** `http://localhost/hotel/test_connection.php`

---

## 🚀 Common Scenarios & Solutions

### Scenario 1: Fresh MySQL Installation
```php
define('DB_HOST', 'localhost');
define('DB_USER', 'root');
define('DB_PASS', '');           // No password
define('DB_NAME', 'hotel_db');
```

### Scenario 2: MySQL with Password Set
```php
define('DB_HOST', 'localhost');
define('DB_USER', 'root');
define('DB_PASS', 'your_mysql_password');
define('DB_NAME', 'hotel_db');
```

### Scenario 3: Using XAMPP
```php
define('DB_HOST', 'localhost');
define('DB_USER', 'root');
define('DB_PASS', '');
define('DB_NAME', 'hotel_db');
```

### Scenario 4: Using WAMP
```php
define('DB_HOST', 'localhost');
define('DB_USER', 'root');
define('DB_PASS', '');
define('DB_NAME', 'hotel_db');
```

### Scenario 5: Using Docker
```php
define('DB_HOST', 'mysql');      // Docker service name
define('DB_USER', 'root');
define('DB_PASS', 'root_password');
define('DB_NAME', 'hotel_db');
```

---

## 🐛 Still Getting Error? Try This:

### 1. Check MySQL Port (Default: 3306)
```php
// Try explicit port
'mysql:host=localhost;port=3306;dbname=hotel_db'
```

### 2. Try 127.0.0.1 Instead of localhost
```php
define('DB_HOST', '127.0.0.1');
```

### 3. Check MySQL Character Set
```php
// Add charset
'mysql:host=localhost;dbname=hotel_db;charset=utf8mb4'
```

### 4. Enable MySQL Error Output (Debug Only)
Edit `includes/config.php`:
```php
// Temporarily add for debugging
ini_set('display_errors', 1);
error_reporting(E_ALL);
```

---

## 📋 Verification Checklist

- [ ] MySQL server is running
- [ ] Database `hotel_db` exists (`SHOW DATABASES;`)
- [ ] Database has all tables (`SHOW TABLES;`)
- [ ] MySQL user has correct credentials
- [ ] `includes/config.php` has correct settings
- [ ] `database/hotel_db.sql` was imported successfully
- [ ] Test file shows "Connection successful"

---

## 🎯 Quick Fix Steps (In Order)

1. **Verify MySQL is running** (Windows Services)
2. **Connect to MySQL:**
   ```cmd
   mysql -u root -p
   ```
3. **Create and import database:**
   ```sql
   CREATE DATABASE hotel_db;
   USE hotel_db;
   -- Import hotel_db.sql contents
   ```
4. **Update `includes/config.php`** with correct credentials
5. **Test connection** with test_connection.php

---

## 💡 Pro Tips

- **Save your MySQL credentials securely** in your notes
- **Never commit passwords** to version control
- **Use environment variables** for passwords in production
- **Always backup your database** before major changes

---

## 🆘 Still Having Issues?

Try this diagnostic command:

```cmd
mysql -u root -h localhost -e "SELECT VERSION();"
```

If this works, MySQL is running. If not:
- MySQL service is down
- MySQL not installed
- Wrong credentials

---

## 📞 Alternative: Use SQLite (No MySQL)

If you can't get MySQL working, contact support for SQLite version of the schema.

---

**Good luck! 🚀 You'll get it working!**
