// Admin Dashboard JavaScript Functions

// Auto-hide alerts after 5 seconds
document.addEventListener('DOMContentLoaded', function() {
    const alerts = document.querySelectorAll('.alert');
    alerts.forEach(alert => {
        setTimeout(() => {
            alert.style.transition = 'opacity 0.3s ease';
            alert.style.opacity = '0';
            setTimeout(() => {
                alert.remove();
            }, 300);
        }, 5000);
    });
});

// Confirm delete action
function confirmDelete(message = 'Are you sure you want to delete this item?') {
    return confirm(message);
}

// Format currency
function formatCurrency(amount) {
    return '$' + parseFloat(amount).toFixed(2).replace(/\B(?=(\d{3})+(?!\d))/g, ',');
}

// Format date
function formatDate(dateString) {
    const date = new Date(dateString);
    return date.toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' });
}

// Show loading spinner
function showLoading(element) {
    if (element) {
        element.innerHTML = '<span class="spinner-border spinner-border-sm mr-2"></span>Loading...';
        element.disabled = true;
    }
}

// Hide loading spinner
function hideLoading(element, text) {
    if (element) {
        element.innerHTML = text;
        element.disabled = false;
    }
}

// AJAX form submission
function submitFormAjax(formId, action) {
    const form = document.getElementById(formId);
    if (!form) return;

    const formData = new FormData(form);
    const submitBtn = form.querySelector('button[type="submit"]');

    showLoading(submitBtn);

    fetch(action, {
        method: 'POST',
        body: formData,
        headers: {
            'X-Requested-With': 'XMLHttpRequest'
        }
    })
    .then(response => response.json())
    .then(data => {
        hideLoading(submitBtn, 'Submit');
        
        if (data.success) {
            showAlert('Success', data.message, 'success');
            if (data.redirect) {
                setTimeout(() => {
                    window.location.href = data.redirect;
                }, 1500);
            } else {
                form.reset();
            }
        } else {
            showAlert('Error', data.message, 'danger');
        }
    })
    .catch(error => {
        hideLoading(submitBtn, 'Submit');
        showAlert('Error', 'An error occurred. Please try again.', 'danger');
        console.error('Error:', error);
    });
}

// Show alert message
function showAlert(title, message, type = 'info') {
    const alertHtml = `
        <div class="alert alert-${type} alert-dismissible fade show" role="alert">
            <strong>${title}:</strong> ${message}
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
    `;
    
    const container = document.querySelector('.page-content') || document.body;
    const div = document.createElement('div');
    div.innerHTML = alertHtml;
    container.insertBefore(div.firstElementChild, container.firstChild);

    // Auto-hide after 5 seconds
    const alert = container.querySelector('.alert');
    setTimeout(() => {
        if (alert && alert.parentNode) {
            alert.style.transition = 'opacity 0.3s ease';
            alert.style.opacity = '0';
            setTimeout(() => {
                alert.remove();
            }, 300);
        }
    }, 5000);
}

// Delete item via AJAX
function deleteItem(id, endpoint) {
    if (!confirmDelete()) return;

    const formData = new FormData();
    formData.append('id', id);
    formData.append('action', 'delete');

    fetch(endpoint, {
        method: 'POST',
        body: formData,
        headers: {
            'X-Requested-With': 'XMLHttpRequest'
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showAlert('Success', data.message, 'success');
            // Remove the row from table
            const row = document.querySelector(`[data-id="${id}"]`);
            if (row) {
                row.style.transition = 'opacity 0.3s ease';
                row.style.opacity = '0';
                setTimeout(() => {
                    row.remove();
                }, 300);
            }
        } else {
            showAlert('Error', data.message, 'danger');
        }
    })
    .catch(error => {
        showAlert('Error', 'An error occurred. Please try again.', 'danger');
        console.error('Error:', error);
    });
}

// Toggle edit mode for inline editing
function toggleEditMode(id) {
    const viewMode = document.getElementById('view-' + id);
    const editMode = document.getElementById('edit-' + id);

    if (viewMode && editMode) {
        viewMode.style.display = viewMode.style.display === 'none' ? 'block' : 'none';
        editMode.style.display = editMode.style.display === 'none' ? 'block' : 'none';
    }
}

// Save inline edited item
function saveEdit(id, endpoint) {
    const form = document.getElementById('edit-form-' + id);
    if (!form) return;

    const formData = new FormData(form);
    formData.append('id', id);
    formData.append('action', 'update');

    fetch(endpoint, {
        method: 'POST',
        body: formData,
        headers: {
            'X-Requested-With': 'XMLHttpRequest'
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showAlert('Success', data.message, 'success');
            setTimeout(() => {
                location.reload();
            }, 1500);
        } else {
            showAlert('Error', data.message, 'danger');
        }
    })
    .catch(error => {
        showAlert('Error', 'An error occurred. Please try again.', 'danger');
        console.error('Error:', error);
    });
}

// Bulk action handler
function handleBulkAction() {
    const action = document.getElementById('bulk-action').value;
    const checkboxes = document.querySelectorAll('input[name="item[]"]:checked');

    if (!action) {
        showAlert('Warning', 'Please select an action', 'warning');
        return;
    }

    if (checkboxes.length === 0) {
        showAlert('Warning', 'Please select at least one item', 'warning');
        return;
    }

    if (!confirm('Are you sure?')) return;

    const ids = Array.from(checkboxes).map(cb => cb.value);
    const formData = new FormData();
    formData.append('action', action);
    formData.append('ids', JSON.stringify(ids));

    fetch(window.location.href, {
        method: 'POST',
        body: formData,
        headers: {
            'X-Requested-With': 'XMLHttpRequest'
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showAlert('Success', data.message, 'success');
            setTimeout(() => {
                location.reload();
            }, 1500);
        } else {
            showAlert('Error', data.message, 'danger');
        }
    })
    .catch(error => {
        showAlert('Error', 'An error occurred. Please try again.', 'danger');
        console.error('Error:', error);
    });
}

// Select all checkboxes
function selectAllCheckboxes(checked) {
    const checkboxes = document.querySelectorAll('input[name="item[]"]');
    checkboxes.forEach(checkbox => {
        checkbox.checked = checked;
    });
}

// Export table to CSV
function exportTableToCSV(filename = 'export.csv') {
    const table = document.querySelector('.table');
    if (!table) {
        showAlert('Error', 'No table found', 'danger');
        return;
    }

    let csv = [];
    const rows = table.querySelectorAll('tr');

    rows.forEach(row => {
        const cols = row.querySelectorAll('td, th');
        const csvRow = [];
        
        cols.forEach(col => {
            csvRow.push('"' + col.innerText.replace(/"/g, '""') + '"');
        });
        
        csv.push(csvRow.join(','));
    });

    downloadCSV(csv.join('\n'), filename);
}

// Download CSV file
function downloadCSV(csv, filename) {
    const link = document.createElement('a');
    link.href = 'data:text/csv;charset=utf-8,' + encodeURIComponent(csv);
    link.download = filename;
    link.click();
}

// Print table
function printTable() {
    const table = document.querySelector('.table');
    if (!table) {
        showAlert('Error', 'No table found', 'danger');
        return;
    }

    const printWindow = window.open('', '', 'width=800,height=600');
    printWindow.document.write('<html><head><title>Print</title>');
    printWindow.document.write('<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">');
    printWindow.document.write('</head><body>');
    printWindow.document.write(table.outerHTML);
    printWindow.document.write('</body></html>');
    printWindow.document.close();
    printWindow.print();
}
