<?php
require_once __DIR__ . '/../includes/config.php';

// Get room categories
try {
    $stmt = $pdo->query("SELECT * FROM room_categories ORDER BY base_price ASC");
    $categories = $stmt->fetchAll();
} catch (Exception $e) {
    $categories = [];
}

$message = '';
$message_type = 'info';

// Handle booking submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $first_name = sanitizeInput($_POST['first_name'] ?? '');
    $last_name = sanitizeInput($_POST['last_name'] ?? '');
    $email = sanitizeInput($_POST['email'] ?? '');
    $phone = sanitizeInput($_POST['phone'] ?? '');
    $check_in = sanitizeInput($_POST['check_in'] ?? '');
    $check_out = sanitizeInput($_POST['check_out'] ?? '');
    $number_of_guests = intval($_POST['number_of_guests'] ?? 1);
    $number_of_rooms = intval($_POST['number_of_rooms'] ?? 1);
    $category_id = intval($_POST['category_id'] ?? 0);
    $special_requests = sanitizeInput($_POST['special_requests'] ?? '');

    // Validation
    $errors = [];

    if (empty($first_name)) $errors[] = 'First name is required';
    if (empty($last_name)) $errors[] = 'Last name is required';
    if (!validateEmail($email)) $errors[] = 'Valid email is required';
    if (!validatePhone($phone)) $errors[] = 'Valid phone number is required';
    if (!validateDate($check_in)) $errors[] = 'Valid check-in date is required';
    if (!validateDate($check_out)) $errors[] = 'Valid check-out date is required';
    if (strtotime($check_out) <= strtotime($check_in)) $errors[] = 'Check-out date must be after check-in date';
    if ($category_id <= 0) $errors[] = 'Please select a room category';

    if (!empty($errors)) {
        $message = implode('<br>', $errors);
        $message_type = 'danger';
    } else {
        try {
            // Get available rooms
            $available = getAvailableRooms($check_in, $check_out, $category_id);

            if (count($available) < $number_of_rooms) {
                $message = 'Not enough rooms available for your dates. Please select fewer rooms or different dates.';
                $message_type = 'warning';
            } else {
                // Create guest
                $stmt = $pdo->prepare("
                    INSERT INTO guests (first_name, last_name, email, phone)
                    VALUES (?, ?, ?, ?)
                ");
                $stmt->execute([$first_name, $last_name, $email, $phone]);
                $guest_id = dbLastInsertId();

                // Get category info
                $stmt = $pdo->prepare("SELECT * FROM room_categories WHERE id = ?");
                $stmt->execute([$category_id]);
                $category = $stmt->fetch();

                // Calculate price
                $total_price = calculateBookingPrice($check_in, $check_out, $category['base_price'], $number_of_rooms);

                // Create bookings for each room
                $booking_reference = generateBookingReference();
                $room_id = $available[0]['id'];

                $stmt = $pdo->prepare("
                    INSERT INTO bookings (
                        booking_reference, guest_id, room_id, check_in_date, check_out_date,
                        number_of_guests, number_of_rooms, total_price, special_requests, status
                    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 'pending')
                ");
                $stmt->execute([
                    $booking_reference, $guest_id, $room_id, $check_in, $check_out,
                    $number_of_guests, $number_of_rooms, $total_price, $special_requests
                ]);

                $message = 'Booking created successfully! Your booking reference: ' . $booking_reference . ' . Total price: ' . formatCurrency($total_price);
                $message_type = 'success';
            }
        } catch (Exception $e) {
            $message = 'Error creating booking: ' . htmlspecialchars($e->getMessage());
            $message_type = 'danger';
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Book Your Stay - GreenDoors Hotel</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="../assets/css/public-style.css">
    <style>
        :root {
            --primary-color: #4a7c59;
            --secondary-color: #6b9a7a;
        }
        
        .booking-hero {
            background: linear-gradient(135deg, var(--primary-color) 0%, var(--secondary-color) 100%);
            color: white;
            padding: 60px 0 80px;
            text-align: center;
        }
        
        .booking-form-container {
            background: white;
            border-radius: 12px;
            padding: 40px;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.15);
            margin-top: -40px;
            position: relative;
            z-index: 10;
        }
        
        .booking-form-title {
            text-align: center;
            margin-bottom: 30px;
        }
        
        .booking-form-title h2 {
            color: var(--primary-color);
            font-weight: 800;
            margin-bottom: 10px;
        }
        
        .form-section {
            margin-bottom: 30px;
        }
        
        .form-section-title {
            font-weight: 700;
            color: var(--primary-color);
            margin-bottom: 20px;
            font-size: 1.1rem;
            border-bottom: 2px solid var(--secondary-color);
            padding-bottom: 10px;
        }
    </style>
</head>
<body>
    <!-- Navigation -->
    <nav class="navbar navbar-expand-lg navbar-custom sticky-top">
        <div class="container-fluid px-4">
            <a class="navbar-brand fw-bold" href="index.php">
                <i class="fas fa-cube"></i> GreenDoors
            </a>
            <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navbarNav">
                <span class="navbar-toggler-icon"></span>
            </button>
            <div class="collapse navbar-collapse" id="navbarNav">
                <ul class="navbar-nav ms-auto">
                    <li class="nav-item"><a class="nav-link" href="index.php">Home</a></li>
                    <li class="nav-item"><a class="nav-link" href="rooms.php">Services</a></li>
                    <li class="nav-item"><a class="nav-link" href="#events">Events</a></li>
                    <li class="nav-item"><a class="nav-link" href="#blog">Blog</a></li>
                    <li class="nav-item"><a class="nav-link" href="booking.php">Signup</a></li>
                    <li class="nav-item"><a class="nav-link login-btn" href="../admin/login.php">LOGIN</a></li>
                </ul>
            </div>
        </div>
    </nav>

    <!-- Hero Section -->
    <section class="booking-hero">
        <div class="container">
            <h1>Book Your Perfect Stay</h1>
            <p>Find and reserve the ideal room for your next vacation</p>
        </div>
    </section>

    <!-- Booking Form -->
    <section class="py-5">
        <div class="container" style="max-width: 900px;">
            <div class="booking-form-container">
                <div class="booking-form-title">
                    <h2><i class="fas fa-calendar-check"></i> Complete Your Booking</h2>
                    <p class="text-muted">Fill in your details to reserve your room</p>
                </div>

                <?php if (!empty($message)): ?>
                    <div class="alert alert-<?php echo $message_type; ?> alert-dismissible fade show" role="alert">
                        <?php echo $message; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <form method="POST" novalidate>
                    <!-- Guest Information -->
                    <div class="form-section">
                        <h5 class="form-section-title"><i class="fas fa-user"></i> Guest Information</h5>
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">First Name <span class="text-danger">*</span></label>
                                <input type="text" class="form-control" name="first_name" required>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Last Name <span class="text-danger">*</span></label>
                                <input type="text" class="form-control" name="last_name" required>
                            </div>
                        </div>
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Email <span class="text-danger">*</span></label>
                                <input type="email" class="form-control" name="email" required>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Phone Number <span class="text-danger">*</span></label>
                                <input type="tel" class="form-control" name="phone" required>
                            </div>
                        </div>
                    </div>

                    <!-- Booking Details -->
                    <div class="form-section">
                        <h5 class="form-section-title"><i class="fas fa-calendar"></i> Booking Details</h5>
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Check-in Date <span class="text-danger">*</span></label>
                                <input type="date" class="form-control" name="check_in" id="check_in" required>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Check-out Date <span class="text-danger">*</span></label>
                                <input type="date" class="form-control" name="check_out" id="check_out" required>
                            </div>
                        </div>
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Number of Guests <span class="text-danger">*</span></label>
                                <select class="form-control" name="number_of_guests" required>
                                    <option value="">Select number of guests</option>
                                    <option value="1">1 Guest</option>
                                    <option value="2">2 Guests</option>
                                    <option value="3">3 Guests</option>
                                    <option value="4">4 Guests</option>
                                    <option value="5">5+ Guests</option>
                                </select>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Number of Rooms <span class="text-danger">*</span></label>
                                <select class="form-control" name="number_of_rooms" required>
                                    <option value="">Select number of rooms</option>
                                    <option value="1">1 Room</option>
                                    <option value="2">2 Rooms</option>
                                    <option value="3">3 Rooms</option>
                                    <option value="4">4+ Rooms</option>
                                </select>
                            </div>
                        </div>
                    </div>

                    <!-- Room Selection -->
                    <div class="form-section">
                        <h5 class="form-section-title"><i class="fas fa-door-open"></i> Select Room Type</h5>
                        <div class="row">
                            <div class="col-12 mb-3">
                                <label class="form-label">Room Category <span class="text-danger">*</span></label>
                                <select class="form-control" name="category_id" required>
                                    <option value="">Select a room category</option>
                                    <?php foreach ($categories as $cat): ?>
                                        <option value="<?php echo $cat['id']; ?>">
                                            <?php echo htmlspecialchars($cat['category_name']); ?> - $<?php echo $cat['base_price']; ?>/night
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                    </div>

                    <!-- Special Requests -->
                    <div class="form-section">
                        <h5 class="form-section-title"><i class="fas fa-message"></i> Special Requests</h5>
                        <div class="mb-3">
                            <label class="form-label">Any special requests or preferences?</label>
                            <textarea class="form-control" name="special_requests" rows="4" placeholder="e.g., High floor, near window, crib needed, etc."></textarea>
                        </div>
                    </div>

                    <!-- Submit Button -->
                    <div class="d-grid">
                        <button type="submit" class="btn btn-primary btn-lg">
                            <i class="fas fa-check-circle"></i> COMPLETE BOOKING
                        </button>
                    </div>

                    <p class="text-center text-muted small mt-3">
                        <i class="fas fa-lock"></i> Your information is secure and encrypted
                    </p>
                </form>
            </div>
        </div>
    </section>

    <!-- Footer -->
    <footer class="footer-section py-5 mt-5">
        <div class="container">
            <div class="row mb-4">
                <div class="col-md-4">
                    <h5><i class="fas fa-cube"></i> GreenDoors</h5>
                    <p>Your gateway to comfort and convenience</p>
                </div>
                <div class="col-md-8">
                    <div class="row">
                        <div class="col-md-3">
                            <h6>Quick Links</h6>
                            <ul>
                                <li><a href="index.php">Home</a></li>
                                <li><a href="rooms.php">Services</a></li>
                                <li><a href="booking.php">Booking</a></li>
                            </ul>
                        </div>
                        <div class="col-md-3">
                            <h6>Company</h6>
                            <ul>
                                <li><a href="#">About</a></li>
                                <li><a href="#">Contact</a></li>
                                <li><a href="#">Support</a></li>
                            </ul>
                        </div>
                        <div class="col-md-3">
                            <h6>Legal</h6>
                            <ul>
                                <li><a href="#">Privacy</a></li>
                                <li><a href="#">Terms</a></li>
                                <li><a href="#">Cookies</a></li>
                            </ul>
                        </div>
                        <div class="col-md-3">
                            <h6>Follow Us</h6>
                            <div class="social-icons">
                                <a href="#"><i class="fab fa-facebook"></i></a>
                                <a href="#"><i class="fab fa-twitter"></i></a>
                                <a href="#"><i class="fab fa-instagram"></i></a>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <hr class="bg-secondary">
            <p class="text-center small">&copy; 2024 GreenDoors Hotel. All rights reserved.</p>
        </div>
    </footer>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Set minimum dates
        const today = new Date().toISOString().split('T')[0];
        document.getElementById('check_in').min = today;
        document.getElementById('check_in').value = today;
        
        document.getElementById('check_in').addEventListener('change', function() {
            const checkInDate = new Date(this.value);
            checkInDate.setDate(checkInDate.getDate() + 1);
            document.getElementById('check_out').min = checkInDate.toISOString().split('T')[0];
        });
    </script>
</body>
</html>
